import pygame
import math
import random
import time
import os

# --- Constants ---
WINDOWED_WIDTH, WINDOWED_HEIGHT = 800, 600  # Default windowed size
HALF_HEIGHT = None  # Will be set dynamically
FOV = math.pi / 3
NUM_RAYS = 60
MAX_DEPTH = 800
DELTA_ANGLE = FOV / NUM_RAYS
TILE = 64
SCALE = None  # Will be set dynamically

# Map (1 = wall, 0 = empty), 21x32, simplified for open space
MAP = [
    [1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1],
    [1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1],
    [1,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1],
    [1,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,0,0,1],
    [1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,1,0,1],
    [1,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1],
    [1,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,0,0,1],
    [1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1],
    [1,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,1,0,0,0,1],
    [1,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,0,0,1],
    [1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1],
    [1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1],
    [1,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,1,0,1],
    [1,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1],
    [1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1],
    [1,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,0,0,1],
    [1,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1],
    [1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,1],
    [1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,0,0,1],
    [1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1],
    [1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1]
]

# Colors
WHITE = (255,255,255)
BLACK = (0,0,0)
RED = (255,0,0)
GREEN = (0,255,0)
BLUE = (0,0,255)
YELLOW = (255,255,0)
GRAY = (70,70,70)  # Darker gray for walls
RED_TINT = (255, 0, 0, 50)

# --- Pygame setup ---
pygame.init()
pygame.mixer.init(frequency=44100, size=-16, channels=2, buffer=4096)
# Start in windowed mode
screen = pygame.display.set_mode((WINDOWED_WIDTH, WINDOWED_HEIGHT))
WIDTH, HEIGHT = WINDOWED_WIDTH, WINDOWED_HEIGHT
HALF_HEIGHT = HEIGHT // 2
SCALE = WIDTH // NUM_RAYS
is_fullscreen = False

clock = pygame.time.Clock()
font = pygame.font.SysFont('Arial', 24)

# --- Player ---
player = {
    "x": TILE*3.5,
    "y": TILE*3.5,
    "angle": 0,
    "bullets": [],
    "health": 100
}

# --- Enemies ---
enemies = [
    {"x": TILE*10.5, "y": TILE*5.5, "angle": 0, "health": 50},
]

# --- Game variables ---
level = 1
killed_monsters = 0
exit_pos = None
enemy_base_health = 50
enemy_base_speed = 50
spawn_interval = 5
last_spawn_time = time.time()
music_load_error = None
music_error_time = 0
music_file = None

# --- Functions ---
def toggle_fullscreen():
    global screen, WIDTH, HEIGHT, HALF_HEIGHT, SCALE, is_fullscreen
    is_fullscreen = not is_fullscreen
    if is_fullscreen:
        screen = pygame.display.set_mode((0, 0), pygame.FULLSCREEN)
        WIDTH, HEIGHT = screen.get_size()
    else:
        screen = pygame.display.set_mode((WINDOWED_WIDTH, WINDOWED_HEIGHT))
        WIDTH, HEIGHT = WINDOWED_WIDTH, WINDOWED_HEIGHT
    HALF_HEIGHT = HEIGHT // 2
    SCALE = WIDTH // NUM_RAYS
    print(f"Screen mode: {'Fullscreen' if is_fullscreen else 'Windowed'}, Resolution: {WIDTH}x{HEIGHT}")

def play_music(level):
    global music_load_error, music_error_time, music_file
    music_load_error = None
    pygame.mixer.music.stop()
    script_dir = os.path.dirname(os.path.abspath(__file__))
    audio_files = [f for f in os.listdir(script_dir) if f.endswith(('.mp3', '.wav'))]
    
    if not audio_files:
        music_load_error = "No .mp3 or .wav files found in scary game folder"
        print(music_load_error)
        music_error_time = time.time()
        music_file = None
        return
    
    if level == 2:
        selected_file = "static_sound.mp3"
    else:
        selected_file = audio_files[0] if level == 1 else audio_files[min(1, len(audio_files)-1)]
    music_file = selected_file
    full_path = os.path.join(script_dir, selected_file)
    
    try:
        if not os.path.exists(full_path):
            music_load_error = f"Music file '{selected_file}' not found"
            print(music_load_error)
            music_error_time = time.time()
            music_file = None
            return
        if not os.access(full_path, os.R_OK):
            music_load_error = f"Cannot read '{selected_file}': Permission denied"
            print(music_load_error)
            music_error_time = time.time()
            music_file = None
            return
        pygame.mixer.music.load(full_path)
        pygame.mixer.music.set_volume(0.5)
        pygame.mixer.music.play(-1)
        print(f"Playing music: {full_path}")
    except pygame.error as e:
        music_load_error = f"Error loading '{selected_file}': {e}"
        print(music_load_error)
        music_error_time = time.time()
        music_file = None

def is_wall(x, y):
    mx, my = int(x // TILE), int(y // TILE)
    if 0 <= my < len(MAP) and 0 <= mx < len(MAP[0]):
        return MAP[my][mx] == 1
    return True

def move_player(keys, dt):
    speed = 150 * dt
    if keys[pygame.K_w]:
        print("W pressed")
        nx = player["x"] + math.cos(player["angle"]) * speed
        ny = player["y"] + math.sin(player["angle"]) * speed
        if not (is_wall(nx + TILE/4, ny) or is_wall(nx - TILE/4, ny) or
                is_wall(nx, ny + TILE/4) or is_wall(nx, ny - TILE/4)):
            player["x"], player["y"] = nx, ny
            print(f"Player pos: {player['x']:.2f}, {player['y']:.2f}")
    if keys[pygame.K_s]:
        print("S pressed")
        nx = player["x"] - math.cos(player["angle"]) * speed
        ny = player["y"] - math.sin(player["angle"]) * speed
        if not (is_wall(nx + TILE/4, ny) or is_wall(nx - TILE/4, ny) or
                is_wall(nx, ny + TILE/4) or is_wall(nx, ny - TILE/4)):
            player["x"], player["y"] = nx, ny
            print(f"Player pos: {player['x']:.2f}, {player['y']:.2f}")
    if keys[pygame.K_a]:
        print("A pressed")
        player["angle"] -= 2 * dt
    if keys[pygame.K_d]:
        print("D pressed")
        player["angle"] += 2 * dt

    global level, killed_monsters, exit_pos, enemy_base_health, enemy_base_speed, spawn_interval
    if exit_pos:
        dx = exit_pos["x"] - player["x"]
        dy = exit_pos["y"] - player["y"]
        dist = math.hypot(dx, dy)
        if dist < TILE / 2:
            level += 1
            killed_monsters = 0
            exit_pos = None
            enemy_base_health += 10
            enemy_base_speed += 10
            spawn_interval = max(1, spawn_interval - 0.5)
            play_music(level)

def update_bullets(dt):
    global killed_monsters
    speed = 300 * dt
    for bullet in player["bullets"][:]:
        bullet["x"] += math.cos(bullet["angle"]) * speed
        bullet["y"] += math.sin(bullet["angle"]) * speed
        if int(bullet["x"]) % 10 == 0 or int(bullet["y"]) % 10 == 0:
            if is_wall(bullet["x"], bullet["y"]):
                player["bullets"].remove(bullet)
                continue
        for enemy in enemies[:]:
            if abs(bullet["x"]-enemy["x"]) < TILE/2 and abs(bullet["y"]-enemy["y"]) < TILE/2:
                enemy["health"] -= 25
                if bullet in player["bullets"]:
                    player["bullets"].remove(bullet)
                if enemy["health"] <= 0:
                    enemies.remove(enemy)
                    killed_monsters += 1

def update_enemies(dt):
    speed = (enemy_base_speed + (level - 1) * 10) * dt
    for enemy in enemies:
        dx = player["x"] - enemy["x"]
        dy = player["y"] - enemy["y"]
        distance = math.hypot(dx, dy)
        if distance > 10:
            nx = enemy["x"] + dx/distance * speed
            ny = enemy["y"] + dy/distance * speed
            if not is_wall(nx, ny):
                enemy["x"], enemy["y"] = nx, ny
        if distance < TILE / 2:
            player["health"] -= 5 * dt

def spawn_enemy():
    if len(enemies) >= 10:
        return
    free_tiles = [(x, y) for y in range(1, len(MAP)-1) for x in range(1, len(MAP[0])-1) if MAP[y][x]==0]
    if free_tiles:
        tile_x, tile_y = random.choice(free_tiles)
        ex = tile_x * TILE + TILE/2
        ey = tile_y * TILE + TILE/2
        health = enemy_base_health + (level - 1) * 10
        enemies.append({"x": ex, "y": ey, "angle": 0, "health": health})

def spawn_exit():
    global exit_pos
    free_tiles = [(x, y) for y in range(1, len(MAP)-1) for x in range(1, len(MAP[0])-1) if MAP[y][x]==0]
    if free_tiles:
        tile_x, tile_y = random.choice(free_tiles)
        ex = tile_x * TILE + TILE/2
        ey = tile_y * TILE + TILE/2
        exit_pos = {"x": ex, "y": ey}

def ray_casting():
    rays = []
    start_angle = player["angle"] - FOV/2
    for ray in range(NUM_RAYS):
        angle = start_angle + ray * DELTA_ANGLE
        sin_a = math.sin(angle)
        cos_a = math.cos(angle)
        for depth in range(0, MAX_DEPTH, 1):
            x = player["x"] + cos_a * depth
            y = player["y"] + sin_a * depth
            if is_wall(x, y):
                rays.append((depth, angle))
                break
        else:
            rays.append((MAX_DEPTH, angle))
    return rays

def draw_3d():
    screen.fill(BLACK)
    rays = ray_casting()
    for ray, (depth, angle) in enumerate(rays):
        depth *= math.cos(player["angle"] - angle)
        height = 21000 / (depth + 0.0001)
        color = GRAY if depth < MAX_DEPTH else BLACK
        pygame.draw.rect(screen, color, (ray*SCALE, HALF_HEIGHT - height//2, SCALE, height))
    
    for bullet in player["bullets"]:
        pygame.draw.line(screen, RED, (WIDTH//2, HALF_HEIGHT),
                         (WIDTH//2 + math.cos(bullet["angle"])*15, HALF_HEIGHT + math.sin(bullet["angle"])*15), 3)
    
    for enemy in enemies:
        dx = enemy["x"] - player["x"]
        dy = enemy["y"] - player["y"]
        distance = math.hypot(dx, dy)
        if distance < MAX_DEPTH:
            angle = math.atan2(dy, dx)
            delta = angle - player["angle"]
            while delta < -math.pi:
                delta += 2*math.pi
            while delta > math.pi:
                delta -= 2*math.pi
            if -FOV/2 < delta < FOV/2:
                steps = int(distance // 10)
                visible = True
                for i in range(1, steps):
                    ix = player["x"] + dx*i/steps
                    iy = player["y"] + dy*i/steps
                    if is_wall(ix, iy):
                        visible = False
                        break
                if visible:
                    h = 21000 / (distance + 0.0001)
                    x = (delta + FOV/2)/FOV*WIDTH
                    pygame.draw.rect(screen, GREEN, (x, HALF_HEIGHT - h//2, SCALE, h))

    if exit_pos:
        dx = exit_pos["x"] - player["x"]
        dy = exit_pos["y"] - player["y"]
        distance = math.hypot(dx, dy)
        if distance < MAX_DEPTH:
            angle = math.atan2(dy, dx)
            delta = angle - player["angle"]
            while delta < -math.pi:
                delta += 2*math.pi
            while delta > math.pi:
                delta -= 2*math.pi
            if -FOV/2 < delta < FOV/2:
                steps = int(distance // 10)
                visible = True
                for i in range(1, steps):
                    ix = player["x"] + dx*i/steps
                    iy = player["y"] + dy*i/steps
                    if is_wall(ix, iy):
                        visible = False
                        break
                if visible:
                    h = 21000 / (distance + 0.0001)
                    x = (delta + FOV/2)/FOV*WIDTH
                    pygame.draw.rect(screen, BLUE, (x, HALF_HEIGHT - h//2, SCALE, h))

def draw_ui():
    pygame.draw.rect(screen, RED, (20, 20, player["health"]*2, 20))
    pygame.draw.rect(screen, WHITE, (20, 20, 200, 20), 2)
    bullet_text = font.render(f"Bullets: {len(player['bullets'])}", True, YELLOW)
    screen.blit(bullet_text, (20, 50))
    level_text = font.render(f"Level: {level}", True, YELLOW)
    screen.blit(level_text, (20, 80))
    killed_text = font.render(f"Killed: {killed_monsters}/15", True, YELLOW)
    screen.blit(killed_text, (20, 110))
    if music_load_error and time.time() - music_error_time < 10:
        error_text = font.render(music_load_error, True, RED)
        screen.blit(error_text, (20, 140))
    elif music_file:
        music_text = font.render(f"Playing: {music_file}", True, YELLOW)
        screen.blit(music_text, (20, 140))

    if level >= 2:
        overlay = pygame.Surface((WIDTH, HEIGHT), pygame.SRCALPHA)
        overlay.fill(RED_TINT)
        screen.blit(overlay, (0, 0))

# Start music
play_music(level)

# --- Main loop ---
running = True
while running:
    dt = clock.tick(60)/1000.0
    print(f"FPS: {clock.get_fps():.1f}")
    keys = pygame.key.get_pressed()
    
    for event in pygame.event.get():
        if event.type == pygame.QUIT:
            running = False
        elif event.type == pygame.KEYDOWN:
            if event.key == pygame.K_SPACE:
                player["bullets"].append({"x": player["x"], "y": player["y"], "angle": player["angle"]})
            elif event.key == pygame.K_f:  # Toggle fullscreen with F key
                toggle_fullscreen()

    move_player(keys, dt)
    update_bullets(dt)
    update_enemies(dt)

    if player["health"] <= 0:
        running = False

    current_time = time.time()
    if current_time - last_spawn_time > spawn_interval:
        spawn_enemy()
        last_spawn_time = current_time

    if killed_monsters >= 15 and exit_pos is None:
        spawn_exit()

    draw_3d()
    draw_ui()
    pygame.display.flip()

pygame.quit()